import math
import pygame


class Enemy:
    DETECTION_RANGE = 8.0
    DETECTION_RANGE_SQ = DETECTION_RANGE ** 2

    ATTACK_RANGE = 0.6
    ATTACK_RANGE_SQ = ATTACK_RANGE ** 2
    ATTACK_DAMAGE = 10
    ATTACK_COOLDOWN = 1.0

    HIT_FLASH_TIME = 0.15

    def __init__(self, x, y, sprite=None):
        self.x = float(x)
        self.y = float(y)

        self.max_health = 40
        self.health = self.max_health
        self.speed = 1.2
        self.radius = 0.12

        self.alive = True
        self.state = "idle"

        self.sprite = sprite
        self.base_color = (200, 40, 40)
        self.color = self.base_color

        self.attack_timer = 0.0
        self.hit_flash_timer = 0.0

    @property
    def pos(self):
        return self.x, self.y

    def distance_sq_to(self, player):
        dx = self.x - player.x
        dy = self.y - player.y
        return dx * dx + dy * dy

    # --------------------
    # Damage
    # --------------------
    def take_damage(self, amount):
        if not self.alive:
            return False

        self.health -= amount
        self.hit_flash_timer = self.HIT_FLASH_TIME
        self.color = (255, 255, 255)

        if self.health <= 0:
            self.alive = False
            return True  # Enemy killed

        return False

    # --------------------
    # Update
    # --------------------
    def update(self, player, world_map, dt):
        if not self.alive or not player.alive:
            return

        # Cooldowns
        if self.attack_timer > 0:
            self.attack_timer -= dt

        if self.hit_flash_timer > 0:
            self.hit_flash_timer -= dt
            if self.hit_flash_timer <= 0:
                self.color = self.base_color

        dist_sq = self.distance_sq_to(player)

        if dist_sq <= self.DETECTION_RANGE_SQ:
            self.state = "chase"
        else:
            self.state = "idle"

        if self.state == "chase":
            self.chase_player(player, world_map, dt)

        if dist_sq <= self.ATTACK_RANGE_SQ:
            self.try_attack(player)

    # --------------------
    # Combat
    # --------------------
    def try_attack(self, player):
        if self.attack_timer <= 0:
            player.take_damage(self.ATTACK_DAMAGE)
            self.attack_timer = self.ATTACK_COOLDOWN

    # --------------------
    # Movement
    # --------------------
    def chase_player(self, player, world_map, dt):
        dx = player.x - self.x
        dy = player.y - self.y
        dist = math.hypot(dx, dy)

        if dist < 0.001:
            return

        nx = dx / dist
        ny = dy / dist

        move_x = nx * self.speed * dt
        move_y = ny * self.speed * dt

        self.try_move(move_x, move_y, world_map)
    def respawn(self,bait1,bait2):
        print("should have respawned, mb")
    def try_move(self, dx, dy, world_map):
        if not world_map.check_collision(self.x + dx, self.y + dy, self.radius):
            self.x += dx
            self.y += dy
            return

        if not world_map.check_collision(self.x + dx, self.y, self.radius):
            self.x += dx
        elif not world_map.check_collision(self.x, self.y + dy, self.radius):
            self.y += dy

    # --------------------
    # Rendering
    # --------------------
    def draw_on_minimap(self, screen, scale=20):
        if self.alive:
            pygame.draw.circle(
                screen,
                self.color,
                (int(self.x * scale), int(self.y * scale)),
                4
            )

